package de.davidartmann.rosa.fragment;

import android.content.Context;
import android.content.DialogInterface;
import android.graphics.drawable.ColorDrawable;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.v4.app.Fragment;
import android.support.v4.content.ContextCompat;
import android.support.v7.app.ActionBar;
import android.support.v7.app.AlertDialog;
import android.support.v7.widget.RecyclerView;
import android.support.v7.widget.StaggeredGridLayoutManager;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;

import java.util.List;

import de.davidartmann.rosa.R;
import de.davidartmann.rosa.activity.MainListActivity;
import de.davidartmann.rosa.adapter.MainListArchivAdapter;
import de.davidartmann.rosa.db.async.DeleteAllInactivePersonsInDb;
import de.davidartmann.rosa.db.async.FindAllActivePersonsFromDb;
import de.davidartmann.rosa.db.async.UpdatePersonsInDb;
import de.davidartmann.rosa.model.Person;

/**
 * Fragment for the main listview of all inactive persons.
 * Created by david on 06.03.16.
 */
public class MainListArchivFragment extends Fragment
        implements DeleteAllInactivePersonsInDb.IDeleteAllInactivePersonsInDb,
        MainListArchivAdapter.IArchivFragmentLongClickMenu, UpdatePersonsInDb.IUpdatePersonsInDb,
        FindAllActivePersonsFromDb.IFindAllActive {

    private IMainListArchivFragment iMainListArchivFragment;
    private boolean isLongClickActive;
    private List<Person> mPersonsToUnarchive;

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);
        this.iMainListArchivFragment = (IMainListArchivFragment) context;
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container,
                             Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.fragment_main_list, container, false);
        setHasOptionsMenu(true);
        RecyclerView recyclerView =
                (RecyclerView) view.findViewById(R.id.fragment_main_list_recyclerview);
        recyclerView.setHasFixedSize(true);
        recyclerView.setLayoutManager(
                new StaggeredGridLayoutManager(2, StaggeredGridLayoutManager.VERTICAL));
        MainListArchivAdapter mainListArchivAdapter =
                new MainListArchivAdapter(R.layout.listview_archiv_viewholder_cardview,
                        view.getContext(), this);
        recyclerView.setAdapter(mainListArchivAdapter);
        setActionBar();
        isLongClickActive = false;
        return view;
    }

    private void setActionBar() {
        ActionBar actionBar = ((MainListActivity) getActivity()).getSupportActionBar();
        if (actionBar != null) {
            actionBar.setDisplayHomeAsUpEnabled(true);
            actionBar.setHomeButtonEnabled(true);
            actionBar.setTitle(getString(R.string.Archiv));
        }
    }

    @Override
    public void onCreateOptionsMenu(Menu menu, MenuInflater inflater) {
        super.onCreateOptionsMenu(menu, inflater);
        menu.clear();
        if (isLongClickActive) {
            inflater.inflate(R.menu.menu_main_list_archiv_longclick, menu);
            setActionBarBgColor(true);
        } else {
            inflater.inflate(R.menu.menu_main_list_archiv, menu);
            setActionBarBgColor(false);
        }
    }

    private void setActionBarBgColor(boolean b) {
        ActionBar actionBar = ((MainListActivity) getActivity()).getSupportActionBar();
        if (actionBar != null) {
            if (b) {
                actionBar.setBackgroundDrawable(new ColorDrawable(
                        ContextCompat.getColor(getContext(), R.color.colorDarkGray)));
            } else {
                actionBar.setBackgroundDrawable(new ColorDrawable(
                        ContextCompat.getColor(getContext(), R.color.colorPrimary)));
            }
        }
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        switch (item.getItemId()) {
            case R.id.menu_main_list_archiv_delete_sweep:
                showDeleteAllPermanentlyDialog();
                return true;
            case R.id.menu_main_list_archiv_unarchive:
                showUnarchiveChoosenPersonDialog();
                return true;
        }
        return super.onOptionsItemSelected(item);
    }

    private void showUnarchiveChoosenPersonDialog() {
        final Context context = getContext();
        AlertDialog.Builder builder = new AlertDialog.Builder(context);
        builder.setMessage(context.getString(R.string.dialog_choosen_person_unarchive));
        builder.setPositiveButton(context.getString(R.string.Ja), new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                new FindAllActivePersonsFromDb(context, MainListArchivFragment.this).execute();
                dialog.dismiss();
            }
        });
        builder.setNegativeButton(context.getString(R.string.Nein), new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                dialog.dismiss();
            }
        });
        builder.show();
    }

    private void showDeleteAllPermanentlyDialog() {
        final Context context = getContext();
        AlertDialog.Builder builder = new AlertDialog.Builder(context);
        builder.setMessage(context.getString(R.string.dialog_delete_archive));
        builder.setPositiveButton(context.getString(R.string.Ja), new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                new DeleteAllInactivePersonsInDb(context, MainListArchivFragment.this).execute();
                dialog.dismiss();
            }
        });
        builder.setNegativeButton(context.getString(R.string.Nein), new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                dialog.dismiss();
            }
        });
        builder.show();
    }

    @Override
    public void onDeletedAllInactive(Integer numberOfDeletedPersons) {
        iMainListArchivFragment.onAllInactivePersonsDeleted(numberOfDeletedPersons);
    }

    @Override
    public void onLongClicked(boolean b, List<Person> personsToUnarchive) {
        mPersonsToUnarchive = personsToUnarchive;
        if (b) {
            isLongClickActive = true;
            getActivity().invalidateOptionsMenu();
        } else {
            isLongClickActive = false;
            getActivity().invalidateOptionsMenu();
        }
    }

    @Override
    public void onUpdate(List<Person> updatedPersons) {
        iMainListArchivFragment.onChoosenPersonsUnarchived();
    }

    @Override
    public void onFindAllActive(List<Person> persons) {
        int numberOfActivePersons = persons.size();
        for (Person p : mPersonsToUnarchive) {
            numberOfActivePersons++;
            p.setPosition(numberOfActivePersons);
        }
        new UpdatePersonsInDb(getContext(), MainListArchivFragment.this, mPersonsToUnarchive)
                .execute();
    }

    public interface IMainListArchivFragment {
        void onAllInactivePersonsDeleted(Integer numberOfDeletedPersons);
        void onChoosenPersonsUnarchived();
    }
}
